// Hammer.js
import { world, system, EquipmentSlot, EntityEquippableComponent } from "@minecraft/server";
import { Durability } from "./ToolDurability.js";
import { LavaResistance, NETHERITE_HAMMER_ID } from "./LavaResistance.js";
import { breakBlockWithSilkTouch, breakBlockWithFortune, checkHammerEnchantments } from "./Enchantment.js";

export const HAMMER_IDENTIFIER = Object.freeze([
    "hammertool:wooden_hammer",
    "hammertool:stone_hammer",
    "hammertool:iron_hammer",
    "hammertool:golden_hammer",
    "hammertool:diamond_hammer",
    NETHERITE_HAMMER_ID,
]);

const hammerDestroyListTags = new Set(["stone", "metal"]);
const hammerDestroyListIDs = new Set(["minecraft:deepslate", "obsidian", "minecraft:calcite"]);
const orePattern = /_ore|gravel|raw_|_nylium|netherrack|brick|_wall|tuff|basalt|blackstone|sandstone|_stone|_deepslate|quartz|mud|purpur|prismarine/;

world.beforeEvents.playerBreakBlock.subscribe(async (data) => {
    const player = data.player;
    const block = data.block;
    const equippableComponent = player.getComponent(EntityEquippableComponent.componentId);
    const item = equippableComponent.getEquipment(EquipmentSlot.Mainhand);
    if (item && HAMMER_IDENTIFIER.includes(item.typeId)) {
        let blocksDestroyed = player.isSneaking ? await useHammer(player, block, item) : 1;
        const maxDurability = item.getComponent('durability').maxDurability;
        blocksDestroyed = Math.min(blocksDestroyed, maxDurability);
        system.run(() => {
            Durability(player, item, blocksDestroyed);
        });
    }
});

async function useHammer(player, block, item) {
    if (!player || !block) return 0;
    const mainDirection = getMainDirection(player.getViewDirection());
    const positions = new Set(getPositionsToDestroy(block.location, mainDirection));
    positions.delete(`${block.x},${block.y},${block.z}`); // Remove the main block

    const { hasSilkTouch, fortuneLevel } = checkHammerEnchantments(item);
    const breakPromises = Array.from(positions).map(async (pos) => {
        const [x, y, z] = pos.split(',').map(Number);
        const targetBlock = player.dimension.getBlock({ x, y, z });
        if (targetBlock && canDestroyBlock(targetBlock, item)) {
            try {
                hasSilkTouch ? await breakBlockWithSilkTouch(player, targetBlock, { x, y, z }) :
                    await breakBlockWithFortune(player, targetBlock, { x, y, z }, fortuneLevel);
                return 1; // Count as successfully broken
            } catch (error) {
                return 0;
            }
        }
        return 0;
    });

    const results = await Promise.all(breakPromises);
    const blocksDestroyed = results.reduce((sum, value) => sum + value, 1); // Add 1 for the main block

    return blocksDestroyed;
}

function canDestroyBlock(block, item) {
    if (block.type.id === "minecraft:obsidian") {
        return item.typeId === "hammertool:diamond_hammer" ||
            item.typeId === NETHERITE_HAMMER_ID ||
            item.hasTag("diamond_pick_diggable");
    }
    for (const tag of block.getTags()) {
        if (hammerDestroyListTags.has(tag)) return true;
    }
    if (hammerDestroyListIDs.has(block.type.id)) return true;
    const itemStack = block.getItemStack();
    if (itemStack && orePattern.test(itemStack.typeId)) return true;
    return false;
}

function getMainDirection(viewDirection) {
    const absX = Math.abs(viewDirection.x);
    const absY = Math.abs(viewDirection.y);
    const absZ = Math.abs(viewDirection.z);
    if (absX > absY && absX > absZ) return viewDirection.x > 0 ? "east" : "west";
    if (absZ > absX && absZ > absY) return viewDirection.z > 0 ? "south" : "north";
    return viewDirection.y > 0 ? "up" : "down";
}

function* getPositionsToDestroy(center, mainDirection) {
    const x = Math.floor(center.x);
    const y = Math.floor(center.y);
    const z = Math.floor(center.z);
    for (let dx = -1; dx <= 1; dx++) {
        for (let dy = -1; dy <= 1; dy++) {
            for (let dz = -1; dz <= 1; dz++) {
                switch (mainDirection) {
                    case "north":
                    case "south":
                        yield `${x + dx},${y + dy},${z}`;
                        break;
                    case "east":
                    case "west":
                        yield `${x},${y + dy},${z + dz}`;
                        break;
                    case "up":
                    case "down":
                        yield `${x + dx},${y},${z + dz}`;
                        break;
                }
            }
        }
    }
}

system.run(() => {
    LavaResistance();
});