


// Reference : https://www.shadertoy.com/view/ldcSRN
//           : http://filmicworlds.com/blog/filmic-tonemapping-operators/
// 
#define SATURATION 1.5

#define Tonemap2Uncharted 0
#define ACESfitted 0
#define Reinhard 1
#define Filmic 0
#define Exposure 4.0

float3 saturation( float3 rgb, float adjustment)
{
    // Algorithm from Chapter 16 of OpenGL Shading Language
    float3 W = float3(0.2125, 0.7154, 0.0721);
    float gray = dot(rgb, W);
    float3 intensity = float3( gray, gray, gray );
    return lerp(intensity, rgb, adjustment);
}

float3 Uncharted2Tonemap(float3 x)
{
	float A = 0.15;
	float B = 0.50;
	float C = 0.10;
	float D = 0.20;
	float E = 0.02;
	float F = 0.30;
	
	return ((x*(A*x+C*B)+D*E)/(x*(A*x+B)+D*F))-E/F;
}

float3 tonemap2uncharted( float3 texColor )
{
	float W = 11.2;
	float ExposureBias = 2.0;
	
	texColor *= 4.0;  // Hardcoded Exposure Adjustment
	float3 curr = Uncharted2Tonemap(ExposureBias*texColor);

	float3 whiteScale = 1.0/Uncharted2Tonemap(float3(W, W, W));
	float3 color = curr*whiteScale;
	
	float spd = 1.0/2.2;
	float3 retColor = pow(color, float3( spd, spd, spd ));
	return retColor;
}

float3 RRTAndODTFit(float3 v)
{
    float3 a = v * (v + 0.0245786) - 0.000090537;
    float3 b = v * (0.983729 * v + 0.4329510) + 0.238081;
    return a / b;
}

float3 ACESFitted(float3 color)
{

	float3x3 ACESInputMat = float3x3(
	    0.59719, 0.35458, 0.04823,
	    0.07600, 0.90834, 0.01566,
	    0.02840, 0.13383, 0.83777
	);

	float3x3 ACESOutputMat = float3x3(
	     1.60475, -0.53108, -0.07367,
	    -0.10208,  1.10813, -0.00605,
	    -0.00327, -0.07276,  1.07602
	);

	color *= Exposure;
	color = pow(color, float3(0.833, 0.833, 0.833)) * 1.07;
    color = mul(color, ACESInputMat);
	color = RRTAndODTFit(color);
	color = mul(color, ACESOutputMat);
	color = clamp(color, 0.0, 1.0);

    return color;
}

float filmic_curve(float x) {
	return ((x*(0.22*x+0.1*0.3)+0.2*0.01)/(x*(0.22*x+0.3)+0.2*0.3))-0.01/0.3;
}

float3 filmic(float3 x) {
    float W = 11.2;
    float w = filmic_curve(W);
    
    x *= Exposure;
    return float3(
        filmic_curve(x.r),
        filmic_curve(x.g),
        filmic_curve(x.b)) / w;
}

float reinhard_curve(float x) {
	return x / (1.0 + x);
}

float3 reinhard(float3 x) {
	float W = 11.2;
    float w = reinhard_curve(W);

    x *= Exposure;
    return float3(
        reinhard_curve(x.r),
        reinhard_curve(x.g),
        reinhard_curve(x.b)) / w;
}

void setColor( inout float3 c ){

	c = max( c * 1.3 - 0.1, float3( 0.0, 0.0, 0.0 ) );

	#if Tonemap2Uncharted
		c = tonemap2uncharted( c );
	#endif

	#if ACESfitted
		c = ACESFitted( c );
	#endif

	#if Reinhard
		c = reinhard( c );
	#endif

	#if Filmic
		c = filmic( c );
	#endif
		
	c = saturation( c, SATURATION );
}